/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999, 2000, 2001  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/
 
#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_ndsvideo.c,v 1.2 2001/07/02 00:20:29 hercules Exp $";
#endif

/* Dummy SDL video driver implementation; this is just enough to make an
 *  SDL-based application THINK it's got a working video driver, for
 *  applications that call SDL_Init(SDL_INIT_VIDEO) when they don't need it,
 *  and also for use as a collection of stubs when porting SDL to a new
 *  platform for which you haven't yet written a valid video driver.
 *
 * This is also a great way to determine bottlenecks: if you think that SDL
 *  is a performance problem for a given platform, enable this driver, and
 *  then see if your application runs faster without video overhead.
 *
 * Initial work by Ryan C. Gordon (icculus@linuxgames.com). A good portion
 *  of this was cut-and-pasted from Stephane Peter's work in the AAlib
 *  SDL video driver.  Renamed to "DUMMY" by Sam Lantinga.
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <nds.h>
#include <nds/registers_alt.h>

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_sysvideo.h"
#include "SDL_pixels_c.h"
#include "SDL_events_c.h"

#include "SDL_ndsvideo.h"
#include "SDL_ndsevents_c.h"
#include "SDL_ndsmouse_c.h"

#define NDSVID_DRIVER_NAME "nds"

/* If defined, always use the largest bitmap size possible for the given mode;
 * If commented out, use the smallest bitmap size that will fit our desired resolution
 */
//#define FORCE_MAX_RESOLUTION

/* Set this to a magic height value, such that we'll vertically center and truncate
 * the display if the application requests a resolution at that height, even if
 * SDL_NDS_STRETCH is passed in as a flag.  "200" is a good value here, since it's
 * kind of silly to scale from 200 to 192 pixels.  Comment out to disable this special-case.
 */
#define MAGIC_HEIGHT 200


/* Initialization/Query functions */
static int NDS_VideoInit(_THIS, SDL_PixelFormat *vformat);
static SDL_Rect **NDS_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags);
static SDL_Surface *NDS_SetVideoMode(_THIS, SDL_Surface *current, int width, int height, int bpp, Uint32 flags);
static int NDS_SetColors(_THIS, int firstcolor, int ncolors, SDL_Color *colors);
static void NDS_VideoQuit(_THIS);

static int NDS_Available(void);
static void NDS_DeleteDevice(SDL_VideoDevice *device);
static SDL_VideoDevice *NDS_CreateDevice(int devindex);

/* Hardware surface functions */
static int NDS_AllocHWSurface(_THIS, SDL_Surface *surface);
static int NDS_LockHWSurface(_THIS, SDL_Surface *surface);
static int NDS_FlipHWSurface(_THIS, SDL_Surface *surface);
static void NDS_UnlockHWSurface(_THIS, SDL_Surface *surface);
static void NDS_FreeHWSurface(_THIS, SDL_Surface *surface);

/* etc. */
static void NDS_UpdateRects(_THIS, int numrects, SDL_Rect *rects);

/* globals */
u16* frontBuffer;
u16* backBuffer;
int sdl_desmume_hack = 0;

VideoBootStrap NDS_bootstrap = {
	NDSVID_DRIVER_NAME, "SDL NDS video driver",
	NDS_Available, NDS_CreateDevice
};

// desmume has a bug where dmaCopy() fails on transfers of 128K or larger.
// so, split large transfers into multiple (128K - 4) byte chunks
//#define DMACOPY_BREAK_POINT 128 * 1024 - 1
#define DMACOPY_BREAK_POINT 128 * 1024 - 4
static inline void dmaCopyEmu(const void *src, void *dest, uint32 size)
{
	if(!sdl_desmume_hack) {
		dmaCopy(src, dest, size);
		return;
	}

	uint32 i = 0;
	while(i < size) {
		dmaCopy((u8*)src + i, (u8*)dest + i, (size - i >= DMACOPY_BREAK_POINT ? DMACOPY_BREAK_POINT : size - i));
		i += DMACOPY_BREAK_POINT;
	}
}

static int NDS_Available(void)
{
	return(1);
}

static void NDS_DeleteDevice(SDL_VideoDevice *device)
{
	if(!device->hidden->hwsurface && frontBuffer != NULL) {
		free(frontBuffer);
		frontBuffer = NULL;
	}
	free(device->hidden);
	free(device);
}

u32 nds_frame_counter = 0;
void on_irq_vblank() 
{
	nds_frame_counter++;

	// Disable interrupts
	//REG_IME = 0;
	//scanKeys();

	//  VBLANK_INTR_WAIT_FLAGS |= IRQ_VBLANK; 
	//  REG_IF |= IRQ_VBLANK; 
	//REG_IF = REG_IF;

	// Enable interrupts
	//REG_IME = 1;
}

static int HWAccelBlit(SDL_Surface *src, SDL_Rect *srcrect,
                        SDL_Surface *dst, SDL_Rect *dstrect)
 {
	fprintf(stderr, "HWAccelBLit\n");
	return 0;
 }
 
static int CheckHWBlit(_THIS, SDL_Surface *src, SDL_Surface *dst)
{
	fprintf(stderr, "CheckHWBLit\n");

	if (src->flags  & SDL_SRCALPHA) return false;
	if (src->flags  & SDL_SRCCOLORKEY) return false;
	if (src->flags  & SDL_HWPALETTE ) return false;
	if (dst->flags  & SDL_SRCALPHA) return false;
	if (dst->flags  & SDL_SRCCOLORKEY) return false;
	if (dst->flags  & SDL_HWPALETTE ) return false;

	if (src->format->BitsPerPixel != dst->format->BitsPerPixel) return false;
	if (src->format->BytesPerPixel != dst->format->BytesPerPixel) return false;
	
	src->map->hw_blit = HWAccelBlit;
	return true;
}

static SDL_VideoDevice *NDS_CreateDevice(int devindex)
{
	SDL_VideoDevice *device=0;

	/* Initialize all variables that we clean on shutdown */
	device = (SDL_VideoDevice *)malloc(sizeof(SDL_VideoDevice));
	if ( device ) {
		memset(device, 0, (sizeof *device));
		device->hidden = (struct SDL_PrivateVideoData *)
				malloc((sizeof *device->hidden));
	}
	if ( (device == NULL) || (device->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( device ) {
			free(device);
		}
		return(0);
	} 
	memset(device->hidden, 0, (sizeof *device->hidden));
	frontBuffer = NULL;
	backBuffer = NULL;

	/* Set the function pointers */
	device->VideoInit = NDS_VideoInit;
	device->ListModes = NDS_ListModes;
	device->SetVideoMode = NDS_SetVideoMode;
	device->CreateYUVOverlay = NULL;
	device->SetColors = NDS_SetColors;
	device->UpdateRects = NDS_UpdateRects;
	device->VideoQuit = NDS_VideoQuit;
	device->AllocHWSurface = NDS_AllocHWSurface;
	device->CheckHWBlit = CheckHWBlit;
	device->FillHWRect = NULL;
	device->SetHWColorKey = NULL;
	device->SetHWAlpha = NULL;
	device->LockHWSurface = NDS_LockHWSurface;
	device->UnlockHWSurface = NDS_UnlockHWSurface;
	device->FlipHWSurface = NDS_FlipHWSurface;
	device->FreeHWSurface = NDS_FreeHWSurface;
	device->SetCaption = NULL;
	device->SetIcon = NULL;
	device->IconifyWindow = NULL;
	device->GrabInput = NULL;
	device->GetWMInfo = NULL;
	device->InitOSKeymap = NDS_InitOSKeymap;
	device->PumpEvents = NDS_PumpEvents;
	device->info.blit_hw=1;

	device->free = NDS_DeleteDevice;
	return device;
}

int NDS_VideoInit(_THIS, SDL_PixelFormat *vformat)
{
	//printf("WARNING: You are using the SDL NDS video driver!\n");

	/* Determine the screen depth (use default 16-bit depth) */
	/* we change this during the SDL_SetVideoMode implementation... */
	vformat->BitsPerPixel = 16;	// mode 3
	vformat->BytesPerPixel = 2;
	vformat->Rmask = 0x0000f800;
	vformat->Gmask = 0x000007e0;
	vformat->Bmask = 0x0000001f; 

    powerON(POWER_ALL);
	irqInit();
	irqSet(IRQ_VBLANK, on_irq_vblank); 
	irqEnable(IRQ_VBLANK);

	//set the sub background up for text display
	videoSetModeSub(MODE_0_2D | DISPLAY_BG0_ACTIVE);
	vramSetBankH(VRAM_H_SUB_BG); // we're going to use banks A-D on the main screen, so we're left with H here
    SUB_BG0_CR = BG_MAP_BASE(8);
	BG_PALETTE_SUB[255] = RGB15(31,31,31);
	consoleInitDefault((u16*)SCREEN_BASE_BLOCK_SUB(8), (u16*)CHAR_BASE_BLOCK_SUB(0), 16); 
	//fprintf(stderr, "Console enabled\n");

	frontBuffer = NULL;
	backBuffer = NULL;

	//lcdSwap();

	/* We're done! */
	return(0); 
}

SDL_Rect **NDS_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags)
{
   	 return (SDL_Rect **) -1;
}

SDL_Surface *NDS_SetVideoMode(_THIS, SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	Uint32 Rmask, Gmask, Bmask, Amask; 

	//printf("NDS_SetVideoMode\n");

	if((bpp > 8 && (width > 512 || height > 512)) || width > 1024 || height > 1024 || (width > 512 && height > 512)) {
		fprintf(stderr, "Requested mode not available; max resolution is 1024x512x8 or 512x512x16\n");
		return NULL;
	}

#ifndef MAGIC_HEIGHT
// default to an un-achievable value; disables this feature
#define MAGIC_HEIGHT -1
#endif

	if(bpp > 8) {
		// the only true color mode we've got is 16bpp, RGBA 5551, so that's what you get
		bpp = 16;
 		Rmask = 0x0000001F;
		Gmask = 0x000003E0;
		Bmask = 0x00007C00;
		Amask = 0x00008000;

		videoSetMode(MODE_5_2D | DISPLAY_BG2_ACTIVE); 
		vramSetMainBanks(VRAM_A_MAIN_BG,VRAM_B_MAIN_BG,VRAM_C_MAIN_BG,VRAM_D_MAIN_BG);

#ifndef FORCE_MAX_RESOLUTION
		// use the smallest bitmap we can get away with
		if(width <= 256 && height <= 256) {
			BG2_CR = BG_BMP16_256x256;
			this->hidden->bg_w = 256;
			this->hidden->bg_h = 256;
		} else if(width <= 512 && height <= 256) {
			BG2_CR = BG_BMP16_512x256;
			this->hidden->bg_w = 512;
			this->hidden->bg_h = 256;
		} else
#endif
		{
			BG2_CR = BG_BMP16_512x512;
			this->hidden->bg_w = 512;
			this->hidden->bg_h = 512;
		}

		if(flags & SDL_NDS_TRUNCATE)
			BG2_XDX = (1 << 8);
		else
			BG2_XDX = ((width / 256) << 8) | (width % 256);

		if((flags & SDL_NDS_TRUNCATE) || height == MAGIC_HEIGHT) {
			// special case: always truncate in 200px-high resolutions, because it's silly to scale from 200 to 192.
			if(sdl_desmume_hack)
				BG2_YDY = (1 << (this->hidden->bg_h == 512 ? 7 : 8)); // wrong, but needed in desmume
			else
				BG2_YDY = (1 << 8);
		} else {
			if(sdl_desmume_hack)
				BG2_YDY = ((height << (this->hidden->bg_h == 512 ? 8 : 9)) / 256 * 2 / 3); // wrong, but needed in desmume
			else
				BG2_YDY = ((height / 192) << 8) | ((height % 192) + (height % 192) / 3) ;
		}

        BG2_XDY = 0; 
        BG2_YDX = 0;	
        BG2_CX = 0;
		BG2_CY = (height == MAGIC_HEIGHT ? (4 << 8) : 0); // vertically-center in 200px-high resolutions

		this->hidden->ndsmode = 5;
		this->hidden->bpp = bpp;
	} else if(bpp <= 8) {
		// palettized 8bpp color, here we come
		bpp = 8;
		Rmask = 0x00000000;
		Gmask = 0x00000000; 
		Bmask = 0x00000000;
		Amask = 0x00000000;

		if(width <= 512 && height <= 512) {
			videoSetMode(MODE_5_2D | DISPLAY_BG2_ACTIVE); 
			vramSetMainBanks(VRAM_A_MAIN_BG,VRAM_B_MAIN_BG,VRAM_C_MAIN_BG,VRAM_D_MAIN_BG);

#ifndef FORCE_MAX_RESOLUTION
			// use the smallest bitmap we can get away with
			if(width <= 256 && height <= 256) {
				BG2_CR = BG_BMP8_256x256;
				this->hidden->bg_w = 256;
				this->hidden->bg_h = 256;
			} else if(width <= 512 && height <= 256) {
				BG2_CR = BG_BMP8_512x256;
				this->hidden->bg_w = 512;
				this->hidden->bg_h = 256;
			} else
#endif
		   	{
				BG2_CR = BG_BMP8_512x512;
				this->hidden->bg_w = 512;
				this->hidden->bg_h = 512;
			}

			if(flags & SDL_NDS_TRUNCATE)
				BG2_XDX = (1 << 8);
			else
				BG2_XDX = ((width / 256) << 8) | (width % 256);

			if((flags & SDL_NDS_TRUNCATE) || height == MAGIC_HEIGHT) {
				// special case: always truncate in 200px-high resolutions, because it's silly to scale from 200 to 192.
				if(sdl_desmume_hack)
					BG2_YDY = (1 << (this->hidden->bg_h == 512 ? 7 : 8)); // wrong, but needed in desmume
				else
					BG2_YDY = (1 << 8);
			} else {
				if(sdl_desmume_hack)
					BG2_YDY = ((height << (this->hidden->bg_h == 512 ? 8 : 9)) / 256 * 2 / 3); // wrong, but needed in desmume
				else
					BG2_YDY = ((height / 192) << 8) | ((height % 192) + (height % 192) / 3) ;
			}

			BG2_XDY = 0; 
			BG2_YDX = 0;
			BG2_CX = 0;
			BG2_CY = (height == MAGIC_HEIGHT ? (4 << 8) : 0); // vertically-center in 200px-high resolutions

			this->hidden->ndsmode = 5;
			this->hidden->bpp = bpp;
		} else {
			// our only option for > 512x512 is mode 6, which gives us 1024x512 @ 8bpp
			// Note that desmume doesn't support this properly (screen will likely display garbage)
			videoSetMode(MODE_6_2D | DISPLAY_BG2_ACTIVE);
			vramSetMainBanks(VRAM_A_MAIN_BG,VRAM_B_MAIN_BG,VRAM_C_MAIN_BG,VRAM_D_MAIN_BG);

			if(height > 512) {
				BG2_CR = BG_BMP8_512x1024;
				this->hidden->bg_w = 512;
				this->hidden->bg_h = 1024;
			} else {
				BG2_CR = BG_BMP8_1024x512;
				this->hidden->bg_w = 1024;
				this->hidden->bg_h = 512;
			}

			if(flags & SDL_NDS_TRUNCATE)
				BG2_XDX = (1 << 8);
			else
				BG2_XDX = ((width / 256) << 8) | (width % 256);

			if((flags & SDL_NDS_TRUNCATE) || height == MAGIC_HEIGHT) {
				// special case: always truncate in 200px-high resolutions, because it's silly to scale from 200 to 192.
				// Note that we don't bother with the desmume hack here--it'd be kind of pointless since desmume can't handle mode 6 at all
				BG2_YDY = (1 << 8);
			} else {
				BG2_YDY = ((height / 192) << 8) | ((height % 192) + (height % 192) / 3) ;
			}

			BG2_XDY = 0; 
			BG2_YDX = 0;
			BG2_CX = 0;
			BG2_CY = (height == MAGIC_HEIGHT ? (4 << 8) : 0); // vertically-center in 200px-high resolutions

			this->hidden->ndsmode = 6;
			this->hidden->bpp = bpp;
		}
	}

	// make sure our surface size is at least as big as the NDS screen, or we'll have issues
	if(width < 256) width = 256;
	if(height < 192) height = 192;

	// don't allow hardware surface in 8bpp modes, because the rest of SDL isn't smart enough to avoid 8-bit writes
	if(bpp == 8) flags &= ~SDL_HWSURFACE;

	if(flags & SDL_HWSURFACE) {
		// use VRAM directly if a hardware surface was requested
		frontBuffer = BG_GFX;
		this->hidden->hwsurface = 1;
	} else {
		// allocate a buffer
		frontBuffer = (u16*)malloc(this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);
		if(frontBuffer == NULL) {
			fprintf(stderr, "Unable to allocate memory for software surface (%d)\n", this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);
			return NULL;
		}
		this->hidden->hwsurface = 0;
	}

	this->hidden->buffer = frontBuffer;

 	fprintf(stderr,"Using %dx%d @%dbpp (ndsmode %d)\n", width, height, bpp, this->hidden->ndsmode);

	// clear out our buffer to begin with
	memset(this->hidden->buffer, 0, this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);

	/* Allocate the new pixel format for the screen */
	if ( ! SDL_ReallocFormat(current, bpp, Rmask, Gmask, Bmask, Amask) ) {
		this->hidden->buffer = NULL;
		SDL_SetError("Couldn't allocate new pixel format for requested mode");
		return(NULL);
	}

	/* Set up the new mode framebuffer */
	current->flags = flags | SDL_FULLSCREEN;
	this->hidden->w = current->w = width;
	this->hidden->h = current->h = height;
	current->pixels = frontBuffer;
	current->pitch = this->hidden->bg_w * this->hidden->bpp / 8;

	// XXX: software double-buffering is stupid, so don't do it
	current->flags &= ~SDL_DOUBLEBUF;
/*
	this->hidden->secondbufferallocd = 0;
	if (flags & SDL_DOUBLEBUF) { 
		backBuffer=(u16*)malloc(this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);
		if(backBuffer != NULL) {
			this->hidden->secondbufferallocd = 1;
			current->pixels = backBuffer; 
		} else {
			current->flags &= ~SDL_DOUBLEBUF;
		}
	}
*/

	/* We're done */
	return(current);
}

static int NDS_AllocHWSurface(_THIS, SDL_Surface *surface)
{
	if(this->hidden->secondbufferallocd) {
		//printf("double double buffer alloc\n");
		return -1;
	}

	return(0);
}

static void NDS_FreeHWSurface(_THIS, SDL_Surface *surface)
{
	if(this->hidden->secondbufferallocd && backBuffer != NULL) {
		free(backBuffer);
		this->hidden->secondbufferallocd=0;
	}
}

/* We need to wait for vertical retrace on page flipped displays */
static int NDS_LockHWSurface(_THIS, SDL_Surface *surface)
{
	//printf("NDS_LockHWSurface\n");
	return(0);
}

static void NDS_UnlockHWSurface(_THIS, SDL_Surface *surface)
{
	//printf("NDS_UnlockHWSurface\n");
	return;
}

static int NDS_FlipHWSurface(_THIS, SDL_Surface *surface)
{
	if(this->hidden->secondbufferallocd){
		while(DISP_Y!=192);
	    while(DISP_Y==192); 

		// XXX: this doesn't actually flip, it's just double buffering
		dmaCopyAsynch(backBuffer, frontBuffer, this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);
	}

	return(0);
}

static void NDS_UpdateRects(_THIS, int numrects, SDL_Rect *rects)
{
	if(!this->hidden->hwsurface) {
		// FIXME: this updates the whole thing, ignoring rects
		dmaCopyEmu(frontBuffer, BG_GFX, this->hidden->bg_w * this->hidden->bg_h * this->hidden->bpp / 8);
	}
}

int NDS_SetColors(_THIS, int firstcolor, int ncolors, SDL_Color *colors)
{
	//printf("SetColors\n");
	short r,g,b;
	
	if(this->hidden->bpp > 8)
	{
		fprintf(stderr, "This is not a palettized mode\n");
		return -1;
	}

	int i;
	int j = firstcolor + ncolors;

	for(i = firstcolor; i < j; i++)
	{
		r = colors[i].r >> 3;
		g = colors[i].g >> 3;
		b = colors[i].b >> 3;
		BG_PALETTE[i] = RGB15(r, g, b);
	} 

	return(0);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably UpdateRects.
*/
void NDS_VideoQuit(_THIS)
{
}
