/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999, 2000, 2001  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_ndsevents.c,v 1.1 2001/04/26 16:50:18 hercules Exp $";
#endif

/* Being a nds driver, there's no event stream. We just define stubs for
   most of the API. */
#include <nds.h>
#include <stdlib.h>
#include <string.h>

#ifdef INCLUDE_REBOOTLIB
#include <reboot.h>
#endif

#include "SDL.h"
#include "SDL_sysevents.h"
#include "SDL_events_c.h"
#include "SDL_ndsvideo.h"
#include "SDL_ndsevents_c.h"

static SDLKey keymap[NDS_NUMKEYS];
char keymem[NDS_NUMKEYS];	/* memorize states of buttons */

// touchpad tracking variables
struct touchscreen_tap {
	u32 time;
	s16 x;
	s16 y;
};
static touchPosition lastTouch = {0};
static struct touchscreen_tap this_tap = {0, 0, 0};
static struct touchscreen_tap last_tap = {0, 0, 0};
extern u32 nds_frame_counter; // defined in SDL_ndsvideo.c

#define TOUCH_THRESHOLD 40		/* max drag distance between motion events, before we junk it as measurement error.  in screen pixels (256x192) */
#define CLICK_SPEED 10			/* max time the pen can be down to be considered a "tap".  in frames (1/60 sec) */
#define DOUBLE_CLICK_SPEED 30	/* max time between pen-up event and pen-down event to be considered a "double tap".  in frames (1/60 sec) */
#define CLICK_THRESHOLD 10      /* max drag distance between pen-down event and pen-up event to be considered a "tap".  in SDL_surface pixels */
#define TOUCH_ACCEL_X 1			/* horizontal acceleration factor for  mouse in touchpad mode (no effect in touchscreen mode) */
#define TOUCH_ACCEL_Y 1			/* vertical acceleration factor for mouse in touchpad mode (no effect in touchscreen mode) */

/* Event Handler */
void NDS_PumpEvents(_THIS)
{
	scanKeys();
	int i;
	SDL_keysym keysym;
	keysym.mod = KMOD_NONE;

#ifdef INCLUDE_REBOOTLIB
	// Yes, I'm building a hook for Lick's rebootlib into SDL directly.
	if(keysHeld() & (KEY_L | KEY_R | KEY_SELECT | KEY_START))
		reboot();
#endif

	// Generate keyboard events for button activity (including d-pad)
	for(i = 0;i < NDS_NUMKEYS; i++)
	{
		keysym.scancode = i;
		keysym.sym = keymap[i];
		if(keysHeld() & (1 << i) && !keymem[i])
		{
			keymem[i] = 1;
			//printf("key pressed %d\n",i);
			SDL_PrivateKeyboard(SDL_PRESSED, &keysym);
		}
		if(!(keysHeld() & (1 << i)) && keymem[i])
		{
			keymem[i] = 0;
			//printf("key released %d\n",i);
			SDL_PrivateKeyboard(SDL_RELEASED, &keysym);
		}
	}

	/* see if the primary display is on the top screen */
	if(REG_POWERCNT & POWER_SWAP_LCDS) {
		/* touchpad mode; drag around to move mouse pointer, double-tap to click */
		if(keysHeld() & KEY_TOUCH) {
			// read current touch position, compute delta from previous position (drag distance)
			touchPosition touch = touchReadXY();
			s16 dx = (lastTouch.px == 0 ? 0 : touch.px - lastTouch.px);
			s16 dy = (lastTouch.py == 0 ? 0 : touch.py - lastTouch.py);

			// ignore extreme changes (assume it's measurement error)
			if((dx < TOUCH_THRESHOLD && dx > -TOUCH_THRESHOLD && dy < TOUCH_THRESHOLD && dy > -TOUCH_THRESHOLD) || (lastTouch.px == 0 && lastTouch.py == 0)) {
				// move the mouse accordingly (relative movement)
				dx = dx * TOUCH_ACCEL_X * this->hidden->w / 256;
				dy = dy * TOUCH_ACCEL_Y * this->hidden->h / 192;
				if(dx != 0 || dy != 0) {
					// don't spam SDL with motion events unless we're actually moving
					SDL_PrivateMouseMotion(0, 1, dx, dy);
				}
				lastTouch = touch;
			}

			// is this is the start of a tap (as opposed to continuing to hold the pen down)?
			if(!this_tap.time) {
				int x, y;
				SDL_GetMouseState(&x, &y);

				// if this is a double-tap; tell SDL that the mouse button is now pressed
				if(nds_frame_counter - last_tap.time < DOUBLE_CLICK_SPEED && abs(last_tap.x - x) < CLICK_THRESHOLD && abs(last_tap.y - y) < CLICK_THRESHOLD) {
					SDL_PrivateMouseButton(SDL_PRESSED, 1, 0, 0);
				}

				// record the time the tap started
				this_tap.time = nds_frame_counter;
				this_tap.x = x;
				this_tap.y = y;
			}
		} else {
			if(this_tap.time) {
				// no longer touching the screen. if SDL still thinks the mouse button is down, tell it to let go
				if(SDL_GetMouseState(NULL, NULL)) {
					SDL_PrivateMouseButton(SDL_RELEASED, 1, 0, 0);
				}

				// check if this was a simple tap, as opposed to a drag
				int x, y;
				SDL_GetMouseState(&x, &y);
				if(nds_frame_counter - this_tap.time < CLICK_SPEED && abs(this_tap.x - x) < CLICK_THRESHOLD && abs(this_tap.y - y) < CLICK_THRESHOLD) {
					// this was a tap, record it so we can check for double-tap later
					last_tap.time = nds_frame_counter;
					last_tap.x = x;
					last_tap.y = y;
				} else {
					// this was not a tap, forget our previously recorded tap
					last_tap.time = 0;
					last_tap.x = 0;
					last_tap.y = 0;
				}

				// we're not touching the screen; reset our state
				this_tap.time = 0;
				this_tap.x = 0;
				this_tap.y = 0;
				lastTouch.px = 0;
			   	lastTouch.py = 0;
			}
		}
	} else {
		/* touchscreen mode; touch directly on screen to click (double-tap not supported) */
		if(keysHeld() & KEY_TOUCH) {
			touchPosition touch = touchReadXY();
			if(touch.px != 0 || touch.py != 0) {
				// move the mouse to wherever we just touched (absolute movement)
				SDL_PrivateMouseMotion(0, 0, touch.px * this->hidden->w / 256, touch.py * this->hidden->h / 192);

				// tell SDL that the mouse button is down, if it thinks it's not already
				if(!SDL_GetMouseState(NULL, NULL)) {
					SDL_PrivateMouseButton(SDL_PRESSED, 1, 0, 0);
				}
			}
		} else {
			// not touching the screen. if SDL still thinks the mouse button is down, tell it to let go
			if(SDL_GetMouseState(NULL, NULL)) {
				SDL_PrivateMouseButton(SDL_RELEASED, 1, 0, 0);
			}
		}
	}
}

void NDS_InitOSKeymap(_THIS)
{
	// The order matters! Indices are the bitmasks from KEYPAD_BITS for each button.
	// - Map the d-pad to the arrow keys in the obvious manner
	// - Map XYBA to WASD (respectively; i.e. match the diamond shape)
	// - Map L&R to Q&E
	// - Map Start to Enter, and Select to Spacebar
	memset(keymem,0,NDS_NUMKEYS);
	keymap[0]=SDLK_d; 		// A
	keymap[1]=SDLK_s; 		// B
	keymap[2]=SDLK_SPACE;	// Select
	keymap[3]=SDLK_RETURN;	// Start
	keymap[4]=SDLK_RIGHT;	// Right
	keymap[5]=SDLK_LEFT;	// Left
	keymap[6]=SDLK_UP;		// Up
	keymap[7]=SDLK_DOWN;	// Down
	keymap[8]=SDLK_e; 		// R
	keymap[9]=SDLK_q; 		// L
	keymap[10]=SDLK_w; 		// X
	keymap[11]=SDLK_a; 		// Y
	//keymap[12]=SDLK_z; 	// Touchscreen pen-down
	//keymap[13]=SDLK_x; 	// Lid closed
}

/* end of SDL_gbaevents.c ... */

